// == content.js (v7 - JDTSS SPA 兼容版) ==
// 功能：自动监听反馈抽屉“发送”按钮 + 在交流历史注入“推送选中方案”按钮

let hasBoundListener = false;
let hasInjectedButton = false;

// === 全局高亮样式（只注入一次）===
(function injectGlobalHighlightStyle() {
  if (document.getElementById('dtss-row-highlight-style')) return;
  const style = document.createElement('style');
  style.id = 'dtss-row-highlight-style';
  style.textContent = `
    .dtss2002-selected-row {
      background-color: #e6f7ff !important;
    }
    .dtss2002-selected-row td,
    .dtss2002-selected-row th {
      border-bottom: 2px solid #1890ff !important;
      padding-bottom: 3px !important;
    }
  `;
  document.head.appendChild(style);
})();

// === Toast 提示 ===
function showTemporaryToast(message, type = 'info') {
  let container = document.getElementById('dtss-toast-container');
  if (!container) {
    container = document.createElement('div');
    container.id = 'dtss-toast-container';
    container.style.cssText = `
      position: fixed; top: 20px; right: 20px; z-index: 2147483647; pointer-events: none;
    `;
    document.body.appendChild(container);
  }

  const toast = document.createElement('div');
  toast.textContent = message;
  toast.style.cssText = `
    padding: 8px 12px; margin: 4px 0; 
    background: ${type === 'success' ? '#07c160' : type === 'error' ? '#fa5151' : '#1890ff'};
    color: white; border-radius: 4px; font-size: 12px;
    box-shadow: 0 2px 8px rgba(0,0,0,0.2);
    animation: dtssFadeInOut 4s ease-in-out forwards;
  `;

  if (!document.getElementById('dtss-toast-style')) {
    const style = document.createElement('style');
    style.id = 'dtss-toast-style';
    style.textContent = `
      @keyframes dtssFadeInOut {
        0%   { opacity: 0; transform: translateY(-10px); }
        10%  { opacity: 1; transform: translateY(0); }
        90%  { opacity: 1; transform: translateY(0); }
        100% { opacity: 0; transform: translateY(-10px); }
      }
    `;
    document.head.appendChild(style);
  }

  container.appendChild(toast);
  setTimeout(() => toast.remove(), 4000);
}

// ================= 主监听：抽屉出现 =================
function startObserving() {
  const observer = new MutationObserver((mutations) => {
    for (const mutation of mutations) {
      if (mutation.type === 'attributes') {
        const target = mutation.target;
        if (
          target.nodeType === 1 &&
          (target.classList?.contains('el-drawer') || target.hasAttribute('aria-label'))
        ) {
          if (isDrawerVisible(target)) {
            bindIfNotAlready(target);
          }
        }
      }
    }
  });

  observer.observe(document.body, {
    childList: true,
    subtree: true,
    attributes: true,
    attributeFilter: ['class', 'style', 'hidden']
  });

  const pollInterval = setInterval(() => {
    if (hasBoundListener) return;
    const drawers = document.querySelectorAll('.el-drawer');
    for (const drawer of drawers) {
      if (isDrawerVisible(drawer)) {
        const titleEl = drawer.querySelector('*');
        const titleText = (titleEl?.textContent || '').replace(/\s+/g, '');
        if (titleText.includes('创建反馈单')) {
          bindIfNotAlready(drawer);
          clearInterval(pollInterval);
          break;
        }
      }
    }
  }, 800);
}

function isDrawerVisible(el) {
  if (!el || el.nodeType !== 1) return false;
  const style = window.getComputedStyle(el);
  if (style.display === 'none' || style.visibility === 'hidden' || el.hidden) return false;
  if (el.classList.contains('el-drawer__open') || el.classList.contains('is-open')) return true;
  const rect = el.getBoundingClientRect();
  return rect.width > 0 && rect.height > 0;
}

function bindIfNotAlready(drawer) {
  if (hasBoundListener || drawer.__dtssBound__) return;
  const titleText = (drawer.textContent || '').replace(/\s+/g, '');
  if (!titleText.includes('创建反馈单')) return;

  drawer.__dtssBound__ = true;
  hasBoundListener = true;
  showTemporaryToast('监听页面“发送”按钮...', 'info');
  attachSendButtonListener(drawer);
}

function attachSendButtonListener(drawer) {
  const handleClick = (e) => {
    const button = e.target.closest('button.el-button--primary');
    if (!button) return;
    const span = button.querySelector('span');
    const buttonText = (span ? span.textContent : button.textContent)?.trim();
    if (buttonText === '发 送') {
      const data = extractFormDataFromDrawer(drawer);
      const hasContent = data.vin || data.advisor || data.plate || data.partNo || data.faultDesc;
      if (!hasContent) {
        showTemporaryToast('⚠️ 表单为空', 'error');
        return;
      }
      chrome.runtime.sendMessage({ action: "sendTableMessage", data }, (response) => {
        if (response?.success) {
          showTemporaryToast('✅ 已提交，请在企微群确认', 'success');
        } else {
          showTemporaryToast('❌ 提交失败，请重试', 'error');
          console.error("推送失败:", response?.error);
        }
      });
    }
  };
  drawer.addEventListener('click', handleClick, true);
}

function extractFormDataFromDrawer(drawer) {
  const getField = (id) => drawer.querySelector(`#${id} input, #${id} textarea`)?.value?.trim() || '';
  return {
    vin: getField('bodyNr'),
    advisor: getField('scname'),
    plate: getField('carnum'),
    partNo: getField('gzjnmBj'),
    faultDesc: getField('gzms')
  };
}

// ================= 交流历史：注入“推送选中方案”按钮 =================

function isCommunicationDrawer(node) {
  return node.id === 'download' ||
         node.classList.contains('wrapper') ||
         !!node.querySelector('#communication-history-table');
}

function injectPushButton(drawer) {
  if (drawer.__dtss2002_injected__) return;
  
  let attempts = 0;
  const maxAttempts = 10;
  
  function tryInject() {
    const historySection = drawer.querySelector('#common-row-08');
    if (historySection && historySection.querySelector('.v-table-toper')) {
      drawer.__dtss2002_injected__ = true;
      //doInject(Array.from(historySection.querySelectorAll('tr.v-body--row')), historySection);
	  doInject(historySection.querySelector('.v-table-body') || historySection, historySection);
    } else if (++attempts < maxAttempts) {
      setTimeout(tryInject, 400);
    }
  }
  
  tryInject();
}

let selectedRow = null; // 提升到全局，避免闭包丢失

function doInject(rowsContainer, historySection) {
  const actionsBar = historySection.querySelector('.v-table-toper > .v-actions');
  if (!actionsBar || actionsBar.__dtssHasButton__) return;
  actionsBar.__dtssHasButton__ = true;
  
  // === 新增：查找“处理方案”列索引 ===
  let solutionColumnIndex = -1;
  const headerCells = historySection.querySelectorAll('.v-table-header th');
  headerCells.forEach((th, index) => {
    const text = (th.textContent || '').replace(/\s+/g, '');
    if (/处理方案|解决方案|回复内容|交流内容/i.test(text)) {
      solutionColumnIndex = index;
    }
  });

  if (solutionColumnIndex === -1) {
    console.warn('[JDTSS] 未找到“处理方案”列，回退到第5列');
    solutionColumnIndex = 4; // 回退
  }

  // 创建按钮
  const btn = document.createElement('button');
  btn.textContent = '📤 推送选中方案';
  Object.assign(btn.style, {
    background: '#1890ff',
    color: 'white',
    border: 'none',
    borderRadius: '4px',
    padding: '4px 10px',
    fontSize: '12px',
    marginLeft: '8px',
    cursor: 'pointer',
    verticalAlign: 'middle'
  });

  // 按钮点击逻辑
  btn.onclick = () => {
	  if (!selectedRow) {
		showTemporaryToast('⚠️ 请先点击一行选择处理方案', 'error');
		return;
	  }

	  // ✅ 使用动态列索引
	  const cells = selectedRow.querySelectorAll('td.v-body--column');
	  const targetCell = cells[solutionColumnIndex];
	  let selectedMessage = '';

	  if (targetCell) {
		// 尝试多种方式提取文本
		selectedMessage = targetCell.querySelector('.v-cell')?.textContent?.trim() ||
						  targetCell.textContent?.trim() ||
						  '';
	  }

	  if (!selectedMessage) {
		showTemporaryToast('❌ 未获取到处理方案内容', 'error');
		return;
	  }
    const data = {
      vin: document.querySelector('#bodyNr input')?.getAttribute('title')?.trim() || '',
      plate: document.querySelector('#carnum input')?.value?.trim() || '',
      faultDesc: document.querySelector('#khms textarea')?.value?.trim() || '',
      advisor: getServiceAdvisor(),
      selectedMessage
    };
    showConfirmDialog(data, (finalMessage) => {
      const finalData = { ...data, selectedMessage: finalMessage };
      chrome.runtime.sendMessage({ action: 'sendManualTableMessage', data: finalData }, (res) => {
        if (chrome.runtime.lastError) {
          console.error('sendMessage error:', chrome.runtime.lastError);
          showTemporaryToast('❌ 推送失败：通信错误', 'error');
        } else if (res?.success) {
          showTemporaryToast('✅ 已成功推送', 'success');
        } else {
          showTemporaryToast('❌ 推送失败', 'error');
        }
      });
    });
  };

  actionsBar.appendChild(btn);

  // === 关键修复：动态监听表格行出现 ===
  let rowsObserver = null;
  let retryCount = 0;
  const maxRetries = 15;

  function tryBindRowClicks() {
    const rows = Array.from(rowsContainer.querySelectorAll('tr.v-body--row'));
    if (rows.length > 0) {
      // 清除旧高亮
      rowsContainer.querySelectorAll('.dtss2002-selected-row').forEach(r => r.classList.remove('dtss2002-selected-row'));

      // 绑定点击事件
      rows.forEach(row => {
        if (!row.__dtssClickBound__) {
          row.__dtssClickBound__ = true;
          row.style.cursor = 'pointer';
          row.addEventListener('click', () => {
            rows.forEach(r => r.classList.remove('dtss2002-selected-row'));
            selectedRow = row;
            row.classList.add('dtss2002-selected-row');
          });
        }
      });

      // 停止轮询
      if (rowsObserver) rowsObserver.disconnect();
      return true;
    } else if (++retryCount < maxRetries) {
      setTimeout(tryBindRowClicks, 400);
      return false;
    } else {
      console.warn('[JDTSS] 未能找到交流历史表格行');
      return false;
    }
  }

  // 立即尝试一次
  if (!tryBindRowClicks()) {
    // 如果还没加载出来，用 MutationObserver 监听变化
    rowsObserver = new MutationObserver((mutations) => {
      for (const m of mutations) {
        if (m.type === 'childList' && m.target === rowsContainer) {
          if (tryBindRowClicks()) {
            rowsObserver.disconnect();
            break;
          }
        }
      }
    });
    rowsObserver.observe(rowsContainer, { childList: true, subtree: true });
  }
}

function getServiceAdvisor() {
  const input = document.querySelector('#scname input');
  return input ? (input.value?.trim() || '') : '';
}

// ================= 监听抽屉出现（交流历史）=================

const tableObserver = new MutationObserver((mutations) => {
  for (const mutation of mutations) {
    if (mutation.type === 'childList') {
      for (const node of mutation.addedNodes) {
        if (node.nodeType === 1 && (node.id === 'download' || isCommunicationDrawer(node))) {
          setTimeout(() => injectPushButton(node), 300);
        }
      }
    }
  }
});
tableObserver.observe(document.body, { childList: true, subtree: true });

// ================= 确认弹窗 =================

function showConfirmDialog(data, onConfirm) {
  if (document.getElementById('dtss2002-modal')) return;
  const overlay = document.createElement('div');
  overlay.id = 'dtss2002-modal';
  Object.assign(overlay.style, {
    position: 'fixed', top: 0, left: 0, width: '100vw', height: '100vh',
    backgroundColor: 'rgba(0,0,0,0.5)', zIndex: 99999,
    display: 'flex', alignItems: 'center', justifyContent: 'center'
  });

  const modal = document.createElement('div');
  Object.assign(modal.style, {
    backgroundColor: 'white', padding: '20px', borderRadius: '8px',
    width: '500px', maxWidth: '90vw', maxHeight: '80vh', overflowY: 'auto',
    fontSize: '14px', boxShadow: '0 4px 12px rgba(0,0,0,0.3)'
  });

  modal.innerHTML = `
    <h3 style="margin-top:0; color:#1890ff;">📤 确认推送处理方案</h3>
    <p><strong>VIN:</strong> ${data.vin || '—'}</p>
    <p><strong>车牌:</strong> ${data.plate || '—'}</p>
    <p><strong>故障描述:</strong> ${data.faultDesc || '—'}</p>
    <p><strong>服务顾问:</strong> ${data.advisor || '—'}</p>
    <p><strong>原处理方案:</strong></p>
    <textarea id="dtss2002-edit-message" rows="4" style="width:100%; padding:6px; border:1px solid #ccc; border-radius:4px;">${data.selectedMessage}</textarea>
    <div style="margin-top:16px; text-align:right;">
      <button id="dtss2002-cancel" type="button" style="margin-right:8px; padding:4px 12px;">取消</button>
      <button id="dtss2002-confirm" type="button" style="background:#1890ff; color:white; border:none; padding:4px 12px; border-radius:4px;">确认推送</button>
    </div>
  `;

  overlay.appendChild(modal);
  document.body.appendChild(overlay);

  modal.querySelector('#dtss2002-cancel').onclick = () => document.body.removeChild(overlay);
  modal.querySelector('#dtss2002-confirm').onclick = () => {
    const editedMessage = modal.querySelector('#dtss2002-edit-message').value.trim();
    if (!editedMessage) {
      showTemporaryToast('⚠️ 处理方案不能为空', 'error');
      return;
    }
    document.body.removeChild(overlay);
    onConfirm(editedMessage);
  };
}

// ================= 核心：支持 SPA 路由跳转的初始化 =================

function mainInit() {
  // 重置状态，允许重新注入
  hasBoundListener = false;
  hasInjectedButton = false;

  // 清除旧标记（可选）
  document.querySelectorAll('[__dtssBound__], [__dtss2002_injected__]').forEach(el => {
    delete el.__dtssBound__;
    delete el.__dtss2002_injected__;
  });

  // 启动主监听
  startObserving();

  // 尝试立即注入已存在的交流历史抽屉
  const existing = document.getElementById('download');
  if (existing) {
    injectPushButton(existing);
  }
}

// 判断是否为 JDTSS 页面（按需调整关键词）
function isJDTSSPage() {
  return /jdtss/i.test(window.location.hostname) || /feedback|case|repair/i.test(window.location.pathname);
}

// 监听 SPA 路由变化
let currentUrl = window.location.href;

function onRouteChange() {
  const newUrl = window.location.href;
  if (newUrl !== currentUrl) {
    currentUrl = newUrl;
    if (isJDTSSPage()) {
      console.log('[JDTSS Plugin] 检测到路由变化，重新初始化...');
      setTimeout(mainInit, 300);
    }
  }
}

// 拦截 history API
const originalPush = history.pushState;
const originalReplace = history.replaceState;

history.pushState = function() {
  originalPush.apply(this, arguments);
  onRouteChange();
};

history.replaceState = function() {
  originalReplace.apply(this, arguments);
  onRouteChange();
};

window.addEventListener('popstate', onRouteChange);

// ================= 启动 =================

function runWhenReady() {
  if (!isJDTSSPage()) return;
  console.log('[JDTSS Plugin] 初始化...');
  mainInit();
}

if (document.readyState === 'loading') {
  document.addEventListener('DOMContentLoaded', runWhenReady);
} else {
  runWhenReady();
}